/*************************************************************************

"FreePastry" Peer-to-Peer Application Development Substrate 

Copyright 2002, Rice University. All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

- Redistributions of source code must retain the above copyright
notice, this list of conditions and the following disclaimer.

- Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

- Neither  the name  of Rice  University (RICE) nor  the names  of its
contributors may be  used to endorse or promote  products derived from
this software without specific prior written permission.

This software is provided by RICE and the contributors on an "as is"
basis, without any representations or warranties of any kind, express
or implied including, but not limited to, representations or
warranties of non-infringement, merchantability or fitness for a
particular purpose. In no event shall RICE or contributors be liable
for any direct, indirect, incidental, special, exemplary, or
consequential damages (including, but not limited to, procurement of
substitute goods or services; loss of use, data, or profits; or
business interruption) however caused and on any theory of liability,
whether in contract, strict liability, or tort (including negligence
or otherwise) arising in any way out of the use of this software, even
if advised of the possibility of such damage.

********************************************************************************/

package rice.persistence;
/*
 *  @(#) Cache.java
 *
 *  @author Ansley Post
 *  @author Alan Mislove
 *  @version $Id: Cache.java,v 1.8 2005/03/11 00:58:16 jeffh Exp $
 */
import java.io.*;

import rice.*;
import rice.p2p.commonapi.*;

/**
 * This interface is the abstraction of something which provides a caching
 * service. Implementations should take in parameters specific to the cache
 * algorithm. Two implementations are provided, the LRUCache and GDSCache. This
 * interface extends the Catalog interface, as the cache provides a Catalog
 * service.
 *
 * @version $Id: Cache.java,v 1.8 2005/03/11 00:58:16 jeffh Exp $
 * @author jeffh
 */
public interface Cache extends Catalog {

  /**
   * Caches an object in this storage. This method is non-blocking. If the
   * object has already been stored at the location id, this method has the
   * effect of calling <code>uncachr(id)</code> followed by <code>cache(id, obj)</code>
   * . This method finishes by calling receiveResult() on the provided
   * continuation with whether or not the object was cached. Note that the
   * object may not actually be cached due to the cache replacement policy.
   * Returns <code>True</code> if the cache actaully stores the object, else
   * <code>False</code> (through receiveResult on c).
   *
   * @param id The object's id.
   * @param metadata The object's metdatadata
   * @param obj The object to cache.
   * @param c The command to run once the operation is complete
   */
  public void cache(Id id, Serializable metadata, Serializable obj, Continuation c);

  /**
   * Removes the object from the list of cached objects. This method is
   * non-blocking. If the object was not in the cached list in the first place,
   * nothing happens and <code>False</code> is returned. Returns <code>True</code>
   * if the action succeeds, else <code>False</code> (through receiveResult on
   * c).
   *
   * @param c The command to run once the operation is complete
   * @param id DESCRIBE THE PARAMETER
   */
  public void uncache(Id id, Continuation c);

  /**
   * Returns the maximum size of the cache, in bytes. The result is returned via
   * the receiveResult method on the provided Continuation with an Integer
   * representing the size.
   *
   * @return The MaximumSize value
   */
  public long getMaximumSize();

  /**
   * Sets the maximum size of the cache, in bytes. Setting this value to a
   * smaller value than the current value may result in object being evicted
   * from the cache. Returns the success or failure of the setSize operation
   * (through receiveResult on c).
   *
   * @param size The new maximum size, in bytes, of the cache.
   * @param c The command to run once the operation is complete
   */
  public void setMaximumSize(int size, Continuation c);
}
