/*************************************************************************

"FreePastry" Peer-to-Peer Application Development Substrate 

Copyright 2002, Rice University. All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

- Redistributions of source code must retain the above copyright
notice, this list of conditions and the following disclaimer.

- Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

- Neither  the name  of Rice  University (RICE) nor  the names  of its
contributors may be  used to endorse or promote  products derived from
this software without specific prior written permission.

This software is provided by RICE and the contributors on an "as is"
basis, without any representations or warranties of any kind, express
or implied including, but not limited to, representations or
warranties of non-infringement, merchantability or fitness for a
particular purpose. In no event shall RICE or contributors be liable
for any direct, indirect, incidental, special, exemplary, or
consequential damages (including, but not limited to, procurement of
substitute goods or services; loss of use, data, or profits; or
business interruption) however caused and on any theory of liability,
whether in contract, strict liability, or tort (including negligence
or otherwise) arising in any way out of the use of this software, even
if advised of the possibility of such damage.

********************************************************************************/

package rice.pastry.standard;

import java.util.Hashtable;
import java.util.Random;

import rice.pastry.NodeHandle;
import rice.pastry.NodeSet;
import rice.pastry.PastryNode;
import rice.pastry.leafset.BroadcastLeafSet;
import rice.pastry.leafset.InitiateLeafSetMaintenance;
import rice.pastry.leafset.LeafSet;
import rice.pastry.leafset.LeafSetProtocolAddress;
import rice.pastry.leafset.RequestLeafSet;
import rice.pastry.messaging.Address;
import rice.pastry.messaging.Message;
import rice.pastry.messaging.MessageReceiver;
import rice.pastry.routing.RoutingTable;
import rice.pastry.security.PastrySecurityManager;

/**
 * An implementation of a periodic-style leafset protocol
 *
 * @version $Id: PeriodicLeafSetProtocol.java,v 1.6 2005/05/17 15:46:25 jeffh
 *      Exp $
 * @author Alan Mislove
 */
public class PeriodicLeafSetProtocol implements MessageReceiver {

  /**
   * DESCRIBE THE FIELD
   */
  protected NodeHandle localHandle;
  /**
   * DESCRIBE THE FIELD
   */
  protected PastryNode localNode;
  /**
   * DESCRIBE THE FIELD
   */
  protected PastrySecurityManager security;
  /**
   * DESCRIBE THE FIELD
   */
  protected LeafSet leafSet;
  /**
   * DESCRIBE THE FIELD
   */
  protected RoutingTable routeTable;
  /**
   * DESCRIBE THE FIELD
   */
  protected Random random;

  private Address address;

  /**
   * NodeHandle -> Long remembers the TIME when we received a BLS from that
   * NodeHandle
   */
  protected Hashtable lastTimeReceivedBLS;

  /**
   * DESCRIBE THE FIELD
   */
  public final static boolean verbose = false;

  /**
   * Related to rapidly determining direct neighbor liveness.
   */
  public final static int PING_NEIGHBOR_PERIOD = 20 * 1000;
  /**
   * DESCRIBE THE FIELD
   */
  public final static int CHECK_LIVENESS_PERIOD = PING_NEIGHBOR_PERIOD + (10 * 1000);
  //30*1000;

  /**
   * Builds a periodic leafset protocol
   *
   * @param ln DESCRIBE THE PARAMETER
   * @param local DESCRIBE THE PARAMETER
   * @param sm DESCRIBE THE PARAMETER
   * @param ls DESCRIBE THE PARAMETER
   * @param rt DESCRIBE THE PARAMETER
   */
  public PeriodicLeafSetProtocol(PastryNode ln, NodeHandle local, PastrySecurityManager sm, LeafSet ls, RoutingTable rt) {
    this.localNode = ln;
    this.localHandle = local;
    this.security = sm;
    this.leafSet = ls;
    this.routeTable = rt;
    this.random = new Random();
    this.lastTimeReceivedBLS = new Hashtable();

    // Removed after meeting on 5/5/2005  Don't know if this is always the appropriate policy.
    //leafSet.addObserver(this);
    address = new LeafSetProtocolAddress();
    localNode.scheduleMsgAtFixedRate(new InitiatePingNeighbor(),
      PING_NEIGHBOR_PERIOD, PING_NEIGHBOR_PERIOD);
  }

  /**
   * Gets the address.
   *
   * @return the address.
   */
  public Address getAddress() {
    return address;
  }

  /**
   * Receives messages.
   *
   * @param msg the message.
   */
  public void receiveMessage(Message msg) {
    if (msg instanceof BroadcastLeafSet) {
      // receive a leafset from another node
      BroadcastLeafSet bls = (BroadcastLeafSet) msg;

      lastTimeReceivedBLS.put(bls.from(), new Long(System.currentTimeMillis()));

      // if we have now successfully joined the ring, set the local node ready
      if (bls.type() == BroadcastLeafSet.JoinInitial) {
        // merge the received leaf set into our own
        leafSet.merge(bls.leafSet(), bls.from(), routeTable, security, false, null);

//				localNode.setReady();
        broadcastAll();
      } else {
        // first check for missing entries in their leafset
        NodeSet set = leafSet.neighborSet(Integer.MAX_VALUE);

        // if we find any missing entries, check their liveness
        for (int i = 0; i < set.size(); i++) {
          if (bls.leafSet().test(set.get(i))) {
            set.get(i).checkLiveness();
          }
        }

        // now check for assumed-dead entries in our leafset
        set = bls.leafSet().neighborSet(Integer.MAX_VALUE);

        // if we find any missing entries, check their liveness
        for (int i = 0; i < set.size(); i++) {
          if (!set.get(i).isAlive()) {
            set.get(i).checkLiveness();
          }
        }

        // merge the received leaf set into our own
        leafSet.merge(bls.leafSet(), bls.from(), routeTable, security, false, null);
      }
    } else if (msg instanceof RequestLeafSet) {
      // request for leaf set from a remote node
      RequestLeafSet rls = (RequestLeafSet) msg;

      rls.returnHandle().receiveMessage(new BroadcastLeafSet(localHandle, leafSet, BroadcastLeafSet.Update));
    } else if (msg instanceof InitiateLeafSetMaintenance) {
      // perform leafset maintenance
      NodeSet set = leafSet.neighborSet(Integer.MAX_VALUE);

      if (set.size() > 1) {
        NodeHandle handle = set.get(random.nextInt(set.size() - 1) + 1);
        handle.receiveMessage(new RequestLeafSet(localHandle));
        handle.receiveMessage(new BroadcastLeafSet(localHandle, leafSet, BroadcastLeafSet.Update));

        NodeHandle check = set.get(random.nextInt(set.size() - 1) + 1);
        check.checkLiveness();
      }
    } else if (msg instanceof InitiatePingNeighbor) {
      // IPN every 20 seconds
      NodeHandle left = leafSet.get(-1);
      NodeHandle right = leafSet.get(1);

      // send BLS to left neighbor
      if (left != null) {
        left.receiveMessage(new BroadcastLeafSet(localHandle, leafSet, BroadcastLeafSet.Update));
        left.receiveMessage(new RequestLeafSet(localHandle));
      }
      // see if received BLS within past 30 seconds from right neighbor
      if (right != null) {
        Long time = (Long) lastTimeReceivedBLS.get(right);
        if (time == null ||
          (time.longValue() < (System.currentTimeMillis() - CHECK_LIVENESS_PERIOD))) {
          // else checkLiveness() on right neighbor
          if (verbose) {
            System.out.println("PeriodicLeafSetProtocol: " + System.currentTimeMillis() + " Checking liveness on right neighbor:" + right);
          }
          right.checkLiveness();
        }
      }
      if (left != null) {
        Long time = (Long) lastTimeReceivedBLS.get(left);
        if (time == null ||
          (time.longValue() < (System.currentTimeMillis() - CHECK_LIVENESS_PERIOD))) {
          // else checkLiveness() on left neighbor
          if (verbose) {
            System.out.println("PeriodicLeafSetProtocol: " + System.currentTimeMillis() + " Checking liveness on left neighbor:" + left);
          }
          left.checkLiveness();
        }
      }
    }
  }


  /**
   * Broadcast the leaf set to all members of the local leaf set.
   *
   */
  protected void broadcastAll() {
    BroadcastLeafSet bls = new BroadcastLeafSet(localHandle, leafSet, BroadcastLeafSet.JoinAdvertise);
    NodeSet set = leafSet.neighborSet(Integer.MAX_VALUE);

    for (int i = 1; i < set.size(); i++) {
      set.get(i).receiveMessage(bls);
    }
  }

  /**
   * Used to kill self if leafset shrunk by too much. NOTE: PLSP is not
   * registered as an observer.
   */
//  public void update(Observable arg0, Object arg1) {
//    NodeSetUpdate nsu = (NodeSetUpdate)arg1;
//    if (!nsu.wasAdded()) {
//      if (localNode.isReady() && !leafSet.isComplete() && leafSet.size() < (leafSet.maxSize()/2)) {
//        // kill self
//        System.out.println("PeriodicLeafSetProtocol: "+System.currentTimeMillis()+" Killing self due to leafset collapse. "+leafSet);
//        localNode.resign();
//      }
//    }
//  }
}
