/*************************************************************************

"FreePastry" Peer-to-Peer Application Development Substrate 

Copyright 2002, Rice University. All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

- Redistributions of source code must retain the above copyright
notice, this list of conditions and the following disclaimer.

- Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

- Neither  the name  of Rice  University (RICE) nor  the names  of its
contributors may be  used to endorse or promote  products derived from
this software without specific prior written permission.

This software is provided by RICE and the contributors on an "as is"
basis, without any representations or warranties of any kind, express
or implied including, but not limited to, representations or
warranties of non-infringement, merchantability or fitness for a
particular purpose. In no event shall RICE or contributors be liable
for any direct, indirect, incidental, special, exemplary, or
consequential damages (including, but not limited to, procurement of
substitute goods or services; loss of use, data, or profits; or
business interruption) however caused and on any theory of liability,
whether in contract, strict liability, or tort (including negligence
or otherwise) arising in any way out of the use of this software, even
if advised of the possibility of such damage.

********************************************************************************/
package rice.pastry.socket;

import java.io.*;
import java.net.*;
import java.util.*;

import rice.pastry.*;
import rice.pastry.client.*;
import rice.pastry.dist.*;
import rice.pastry.join.*;
import rice.pastry.leafset.*;
import rice.pastry.messaging.*;
import rice.pastry.routing.*;
import rice.selector.*;

/**
 * An Socket-based Pastry node, which has two threads - one thread for
 * performing route set and leaf set maintainance, and another thread for
 * listening on the sockets and performing all non-blocking I/O.
 *
 * @version $Id: SocketPastryNode.java,v 1.30 2005/05/16 19:11:17 amislove Exp $
 * @author Alan Mislove
 */
public class SocketPastryNode extends DistPastryNode {

  // The address (ip + port) of this pastry node
  private EpochInetSocketAddress address;

  // The SocketManager, controlling the sockets
  private SocketSourceRouteManager srManager;

  // The pool of all node handles
  private SocketNodeHandlePool pool;

  /**
   * DESCRIBE THE FIELD
   */
  public final static boolean verbose = false;

  /**
   * Constructor
   *
   * @param id The NodeId of this Pastry node.
   */
  public SocketPastryNode(NodeId id) {
    super(id);
  }

  /**
   * Returns the SocketSourceRouteManager for this pastry node.
   *
   * @return The SocketSourceRouteManager for this pastry node.
   */
  public SocketSourceRouteManager getSocketSourceRouteManager() {
    return srManager;
  }

  /**
   * Returns the WireNodeHandlePool for this pastry node.
   *
   * @return The WireNodeHandlePool for this pastry node.
   */
  public DistNodeHandlePool getNodeHandlePool() {
    return pool;
  }

  /**
   * Helper method which allows the WirePastryNodeFactory to initialize a number
   * of the pastry node's elements.
   *
   * @param address The address of this pastry node.
   * @param lsmf Leaf set maintenance frequency. 0 means never.
   * @param rsmf Route set maintenance frequency. 0 means never.
   * @param pool The new SocketElements value
   * @param srManager The new SocketElements value
   */
  public void setSocketElements(EpochInetSocketAddress address,
                                SocketSourceRouteManager srManager,
                                SocketNodeHandlePool pool,
                                int lsmf,
                                int rsmf) {
    this.address = address;
    this.srManager = srManager;
    this.pool = pool;
    this.leafSetMaintFreq = lsmf;
    this.routeSetMaintFreq = rsmf;
  }

  /**
   * Called after the node is initialized.
   *
   * @param bootstrap The node which this node should boot off of.
   */
  public void doneNode(NodeHandle bootstrap) {
    super.doneNode(bootstrap);
    initiateJoin(bootstrap);
  }

  /**
   * Prints out a String representation of this node
   *
   * @return a String
   */
  public String toString() {
    return "SocketNodeHandle (" + getNodeId() + "/" + address + ")\n";
  }

  /**
   * Makes this node resign from the network. Is designed to be used for
   * debugging and testing.
   */
  public void resign() {
    try {
      super.resign();
      srManager.resign();
    } catch (IOException e) {
      System.err.println("ERROR: Got exception " + e + " while resigning node!");
    }
//    System.out.println("SocketPastryNode resigning at time:"+System.currentTimeMillis());
//    System.exit(20);
  }
}
